/*

Copyright (c) 2004 PXI Project Team

Permission is hereby granted, free of charge, to any person obtaining a copy of
this software and associated documentation files (the "Software"), to deal in the
Software without restriction, including without limitation the rights to use,
copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
Software, and to permit persons to whom the Software is furnished to do so,
subject to the following conditions:

The above copyright notice and this permission notice shall be included in all copies
or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include "bchasm.h"
#include <iostream>
#include <iomanip>
#include <vector>
#include <cmath>
#include <cstring>
using namespace std;

extern PXIAbstractFramework myAbstractFramework;

bchasm::bchasm()
{
    // Build Null Program
	ip_start = 0;						// Set the instruction pointer to 0.
    bytecode = new int * [ 1 ];			// Allocate enough space for
    bytecode[ 0 ] = new int[ 3 ];		// one instruction.
    bytecode[ 0 ][ 0 ] = 97;			// Set the instruction to HALT
    bytecode[ 0 ][ 1 ] = 0;				// Set the operands to 0.
    bytecode[ 0 ][ 2 ] = 0;				// Set the operands to 0.
    bytecode_length = 1;				// Set the program length.
	cur_proc = "";						// Reset the assembler.
}

bchasm::bchasm( const string & s )
{
	// Build the passed in program.
	ip_start = 0;						// Set the instruction pointer to 0.
	cur_proc = "";						// Reset the assembler
	assemble(s);						// Run the assembler against the incoming code.
}

bchasm::~bchasm()
{
	// Eliminate the evidence. >:
	int c;
	
	// Run through and deallocate every instruction.
	// Isn't C fun?
    for ( c = 0;c < bytecode_length;++c )
    {
        delete [] bytecode[ c ];
    }
	
	// Deallocate the program itself.
	delete [] bytecode;
}

void bchasm::initialize( void )
{
    int c;
    stack_lookup_table.clear();
    label_lookup_table.clear();
    var_lookup_table.clear();

    stack_lookup_indices.clear();
    label_lookup_indices.clear();
    var_lookup_indices.clear();
	
	cur_proc = "";

    // Aaaah! Must put out flaming goo of death!

    for ( c = 0;c < bytecode_length;++c )
    {
        delete [] bytecode[ c ];
    }

    delete [] bytecode;
}

bchasm & bchasm::operator= ( const bchasm & b )
{
	if ( b.source != "" )
	{
		assemble ( b.source );
	}
	return * this;
}

int bchasm::bchasm_handle_softinterrupt ( int interrupt_id, int stack_pointer )
{
	char buffer[ 1024 ] = { 0 };
	string s,s2,s3,s4;
	double d,d2;
	variant v,v2;
	switch ( interrupt_id )
	{
		case 0:
			// Terminate immediately.
			exit( 1 );
			break;
		case 1:
			// stdin
			cout << stack_lookup_table[ stack_pointer ].top().strVal() << endl;
			stack_lookup_table[ stack_pointer ].pop();
			cin.getline( buffer, 1024, '\n' );
			stack_lookup_table[ stack_pointer ].push( string( buffer ) );
			return 0;
			break;
		case 2:
			// stdout
			//cout << stack_lookup_table[ stack_pointer ].top().strVal() << endl;
			standard_out += stack_lookup_table[ stack_pointer ].top().strVal() + "\n";
			stack_lookup_table[ stack_pointer ].pop();
			return 0;
			break;
		case 4001:
		    // absolute value
		    d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
		    stack_lookup_table[ stack_pointer ].pop();
		    stack_lookup_table[ stack_pointer].push( ( double ) ( d < 0 ) ? 0 - d : d );
			break;
        case 4002:
			// annuity
			d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			d2 = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) ( ( 1 - ( 1 + d ) - d2 ) / d ) );
			return 0;
			break;
        case 4003:
			// atan
			d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) atan( d ) );
			return 0;
			break;	
        case 4004:
            // Average
		case 4005:
			// chartonum
			s = ( stack_lookup_table[ stack_pointer ].top().strVal() ) [ 0 ];
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) s [ 0 ] );
			return 0;
			break;
		case 4006:
            //clickChunk
        case 4007:
            //clickH
        case 4008:
            //clickV
        case 4009:
            //clickLine
        case 4010:
            //clickLoc
        case 4011:
            //clickText
        case 4012:
            //commandKey
        case 4013:
            //compound  
        case 4014:
            //cos
            d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) cos( d ) );
			return 0;
			break;            
        case 4015:
            //date  
        case 4016:
            //destination
        case 4017:
            //diskSpace
        case 4018:
            //exp
            d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) exp( d ) );
			return 0;
			break;
        case 4019:
            //exp1
            d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) exp( d )-1 );
			return 0;
			break;
        case 4020:
            //exp2
            d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) 2*exp( d ) );
			return 0;
			break;
        case 4021:
            //foundChunk
        case 4022:
            //foundField
        case 4023:
            //foundLine
        case 4024:
            //foundText
        case 4025:
			// length
			s = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) s.size() );
			return 0;
			break;   
        case 4026:
            //ln 
            d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) log( d ) );      
			return 0;
			break;
        case 4027:
            //ln1
            d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) log( d ) );
			return 0;
			break;
        case 4028:
            //log2
            d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) log2( d ) );
			return 0;
			break;
        case 4029:
            //max
        case 4030:
            //min
        case 4031:
            //mouse
        case 4032:
            //mouseClick
        case 4033:
            //mouseH
        case 4034:
            //mouseLoc
        case 4035:  
            //mouseV
        case 4036:
            //number of objects
        case 4037:
            //number of background buttons
        case 4038:
            //number of (card) fields
        case 4039:
            //number of backgrounds in this stack
        case 4040:
            //number of cards
        case 4041:
            //number of cards in background
        case 4042:
            //number of chunks
        case 4043:
            // numtochar
			d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			buffer[ 0 ] = ( char ) d;
			buffer[ 1 ] = 0;
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( variant( buffer ) );
			return 0;
			break;
        case 4044:
            //offset
        case 4045:
           //param
        case 4046:
           //paramCount  
         case 4047:
            //params
		 case 4048:
           //random
			d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) ( rand() % ( int ) d ) + 1 );
			return 0;
			break;
		 case 4049:
           //result
			 stack_lookup_table[ stack_pointer ].push( var_lookup_table[ 0 ] );
			 return 0;
			 break;
         case 4050:
            //round           
		 case 4051:
           //screenRect
		 case 4052:
           //seconds since Jan 1, 1904
			 stack_lookup_table[ stack_pointer ].push( ( double ) time( NULL ) );
			 return 0;
			 break;
         case 4053:
            //selectedButton
		 case 4054:
           //selectedChunk
		 case 4055:
           //selectedField     
         case 4056:
            //selectedLine
		 case 4057:
           //selectedLoc
		 case 4058:
           //selectedText
         case 4059:
            //shiftKey
		 case 4060:
           //sine
           d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) sin( d ) );
			return 0;
			break;
		 case 4061:
           //sound
        case 4062:
           //sqrt      
           d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) sqrt( d ) );
			return 0;
			break;
        case 4063:
           //stack
        case 4064:
           //stackSpace 
        case 4065:
           //systemVersion
        case 4066:
           //tan
			d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) tan( d ) );
			return 0;
			break;
        case 4067:
           //target
        case 4068:
           //ticks
			stack_lookup_table[ stack_pointer ].push( ( double ) ( time(NULL)*60 + clock() ) );
			return 0;
			break;
        case 4069:
           //time
        case 4070:
           //short time
        case 4071:
            //long time
        case 4072:
           //tool
        case 4073:
           //trunc 
        case 4074:
            //value
        case 4075:
           //windows available                     
			break;
			
		/* Experimental block */
		case 5000:
			//sum
			d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			d2 = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) d + d2 );
			return 0;
			break;
		case 5001:
			// count_chunks
			s = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			s2 = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( ( double ) cxl_count_chunks( s, s2 ) );
			return 0;
			break;
			
		case 6000:
			// create window with title
			s = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			myAbstractFramework.createNewWindow( s );
			return 0;
			break;
			
		case 6030:
			// set tool
			d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			myAbstractFramework.setCurrentTool( (int)d );
			return 0;
			break;
			
		case 6050:
			// Ask Dialog
			s2 = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			s = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			s = myAbstractFramework.ask( s, s2 );
			stack_lookup_table[ stack_pointer ].push( s );
			return 0;
			break;

		case 6051:
			// Answer Dialog
			s4 = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			s3 = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			s2 = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			s = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			s = myAbstractFramework.answer( s, s2, s3, s4 );
			stack_lookup_table[ stack_pointer ].push( s );
			return 0;
			break;			
			
		case 7000:
			// get a global variable
			d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( myAbstractFramework.getGlobalVariable( (int)d ) );
			return 0;
			break;
			
		case 7001:
			// set a global variable
			d = ( stack_lookup_table[ stack_pointer ].top().intVal() );
			stack_lookup_table[ stack_pointer ].pop();
			v = ( stack_lookup_table[ stack_pointer ].top().strVal() );
			stack_lookup_table[ stack_pointer ].pop();
			myAbstractFramework.setGlobalVariable( (int)d, v );
			return 0;
			break;
			
		case 7002:
			// Set a data member.
			v = ( stack_lookup_table[ stack_pointer ].top() ); // Value
			stack_lookup_table[ stack_pointer ].pop();
			v2 = ( stack_lookup_table[ stack_pointer ].top() ); // Object
			stack_lookup_table[ stack_pointer ].pop();
			s = ( stack_lookup_table[ stack_pointer ].top().strVal() ); // Property Name
			stack_lookup_table[ stack_pointer ].pop();
			v2.setMember ( s, v );
			stack_lookup_table[ stack_pointer ].push( v2 );
			return 0;
			break;
			
		case 7003:
			// Fetch a data member.
			s = ( stack_lookup_table[ stack_pointer ].top().strVal() ); // Property Name
			stack_lookup_table[ stack_pointer ].pop();
			v = ( stack_lookup_table[ stack_pointer ].top() ); // Object
			stack_lookup_table[ stack_pointer ].pop();
			stack_lookup_table[ stack_pointer ].push( v.getMember( s ) );
			//stack_lookup_table[ stack_pointer ].push( variant () );
			//stack_lookup_table[ stack_pointer ].top().linkTo( v.getMember( s ) );
			return 0;
			break;
	}
	return 1;
}

string bchasm::interpret_with_prams ( const stack< variant > & prams )
{
	stack_lookup_table[ 0 ] = prams; // Initialize stack with parameter stack.
	return interpret();
}

bool bchasm::has_handler( const string & handlername )
{
	return ( label_lookup_indices.find( "h" + handlername ) != label_lookup_indices.end() );
}

bool bchasm::has_function( const string & functionname )
{
	return ( label_lookup_indices.find( "f" + functionname ) != label_lookup_indices.end() );
}

variant	bchasm::call_handler ( const string & handlername, const stack< variant > & prams )
{
	stack_lookup_table[ 0 ] = prams;
	if( label_lookup_indices.find( "h" + handlername ) != label_lookup_indices.end() )
	{
		// Simulate a <call> instruction
		// 1. Push the instruction pointer onto the stack
		// 2. Set the instruction pointer to the location of the handler.
	
		stack_lookup_table[ 0 ].push( variant( 3 ) ); // Pushes the location of the <jump ,label %xionexit> instruction
		return( variant ( interpret( ( *label_lookup_indices.find( "h" + handlername ) ).second ) ) );
	}
}

variant	bchasm::call_function ( const string & functionname, const stack< variant > & prams )
{
	stack_lookup_table[ 0 ] = prams;
	if( label_lookup_indices.find( "f" + functionname ) != label_lookup_indices.end() )
	{
		// Simulate a <call> instruction
		// 1. Push the instruction pointer onto the stack
		// 2. Set the instruction pointer to the location of the handler.
		
		stack_lookup_table[ 0 ].push( variant( 3 ) ); // Pushes the location of the <jump ,label %xionexit> instruction
		return( variant ( interpret( ( *label_lookup_indices.find( "f" + functionname ) ).second ) ) );
	}
}

string bchasm::interpret( int ip_start )
{
    int c;
    int stack_pointer = 0;
	//ip_start = 0;
    variant v, v2;
    for ( c = ip_start;c < bytecode_length;++c )
    {
        switch ( bytecode[ c ][ 0 ] )
        {
        case 0:
            // <mov> || <move>
            var_lookup_table[ bytecode[ c ][ 2 ] ] = var_lookup_table[ bytecode[ c ][ 1 ] ];
            break;
        case 1:
            // <psh> || <push>
            if ( bytecode[ c ][ 2 ] == -100 )
            {
                stack_lookup_table[ stack_pointer ].push( var_lookup_table[ bytecode[ c ][ 1 ] ] );
            }
            else
            {
                merge_values( stack_lookup_table[ stack_pointer ].top(), var_lookup_table[ bytecode[ c ][ 1 ] ], bytecode[ c ][ 2 ], stack_lookup_table[ stack_pointer ].top() );
            }
            break;
        case 2:
            // <pop>
            merge_values( var_lookup_table[ bytecode[ c ][ 1 ] ], stack_lookup_table[ stack_pointer ].top(), bytecode[ c ][ 2 ], var_lookup_table[ bytecode[ c ][ 1 ] ] );
            stack_lookup_table[ stack_pointer ].pop();
            break;
        case 3:
            // <mrg> || <merge>
            v = stack_lookup_table[ stack_pointer ].top();
            stack_lookup_table[ stack_pointer ].pop();
            merge_values( stack_lookup_table[ stack_pointer ].top(), v, bytecode[ c ][ 2 ], stack_lookup_table[ stack_pointer ].top() );
            break;
        case 4:
            // <swp> || <swap>
            v = stack_lookup_table[ stack_pointer ].top();
            stack_lookup_table[ stack_pointer ].pop();
            v2 = stack_lookup_table[ stack_pointer ].top();
            stack_lookup_table[ stack_pointer ].top() = v;
            stack_lookup_table[ stack_pointer ].push( v2 );
			break;
        case 5:
            // <scs> || <change_stack>
            stack_pointer = bytecode[ c ][ 2 ];
			break;
        case 6:
            // <add>
            //var_lookup_table[bytecode[c][2]] = merge_values(var_lookup_table[bytecode[c][2]],var_lookup_table[bytecode[c][1]],-101);
            var_lookup_table[ bytecode[ c ][ 2 ] ] = var_lookup_table[ bytecode[ c ][ 2 ] ].intVal() + var_lookup_table[ bytecode[ c ][ 1 ] ].intVal();
            break;
        case 7:
            // <sub> || <subtract>
            merge_values( var_lookup_table[ bytecode[ c ][ 2 ] ], var_lookup_table[ bytecode[ c ][ 1 ] ], -102, var_lookup_table[ bytecode[ c ][ 2 ] ] );
            break;
        case 8:
            // <mul> || <multiply>
            merge_values( var_lookup_table[ bytecode[ c ][ 2 ] ], var_lookup_table[ bytecode[ c ][ 1 ] ], -103, var_lookup_table[ bytecode[ c ][ 2 ] ] );
            break;
        case 9:
            // <jmp> || <jump>
            c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            break;
        case 10:
            // <jmt> || <jump_if_mem_true>
            if ( strcmp( var_lookup_table[ bytecode[ c ][ 1 ] ].strVal().c_str(), "true" ) == 0 )
            {
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            }
            break;
        case 11:
            // <jmt> || <jump_if_mem_false>
            if ( strcmp( var_lookup_table[ bytecode[ c ][ 1 ] ].strVal().c_str(), "false" ) == 0 )
            {
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            }
            break;
        case 12:
            // <jmz> || <jump_if_mem_zero>
            if ( var_lookup_table[ bytecode[ c ][ 1 ] ].intVal() == 0 )
            {
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            }
            break;
        case 13:
            // <jmn> || <jump_if_mem_nonzero>
            if ( var_lookup_table[ bytecode[ c ][ 1 ] ].intVal() != 0 )
            {
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            }
            break;
        case 14:
            // <jst> || <jump_if_stack_true>
            if ( strcmp( stack_lookup_table[ stack_pointer ].top().strVal().c_str(), "true" ) == 0 )
            {
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            }
            break;
        case 15:
            // <jsf> || <jump_if_stack_false>
            if ( strcmp( stack_lookup_table[ stack_pointer ].top().strVal().c_str(), "false" ) == 0 )
            {
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            }
            break;
        case 16:
            // <jsz> || <jump_if_stack_zero>
            if ( stack_lookup_table[ stack_pointer ].top().intVal() == 0 )
            {
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            }
            break;
        case 17:
            // <jsn> || <jump_if_stack_nonzero>
            if ( stack_lookup_table[ stack_pointer ].top().intVal() != 0 )
            {
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            }
            break;
		case 18:
			// <jse> || <jump_if_stack_equals>
			if ( strcmpi( var_lookup_table[ bytecode[ c ][ 1 ] ].strVal().c_str(), stack_lookup_table[ stack_pointer ].top().strVal().c_str() ) == 0 )
			{
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
			}
			break;
		case 26:
			// <jump_if_stack_notequals>
			if ( strcmpi( var_lookup_table[ bytecode[ c ][ 1 ] ].strVal().c_str(), stack_lookup_table[ stack_pointer ].top().strVal().c_str() ) != 0 )
			{
                c = label_lookup_table[ bytecode[ c ][ 2 ] ];
			}
			break;
        case 19:
            // <nop> || <skip>
            break;
		case 20:
			// <make>
			switch( bytecode[ c ][ 1 ] )
			{
				case 25:
					// Chunk
					stack_lookup_table[ 1 ].push( variant( 0 ) );
					if( bytecode[ c ][ 2 ] == 100 )
					{
						stack_lookup_table[ 1 ].top().chunkValue = new chunk(false);
					}
					else
					{
						stack_lookup_table[ 1 ].top().chunkValue = new chunk(true);
					}
					break;
			}
			break;
		case 21:
			// <chunk_add>
			( * stack_lookup_table[ 1 ].top().chunkValue ).add_index( static_cast<int>( stack_lookup_table[ stack_pointer ].top().intVal() ), bytecode[ c ][ 2 ] );
			stack_lookup_table[ stack_pointer ].pop();
			break;
		case 22:
			// <chunk_load>
			v = stack_lookup_table[ stack_pointer ].top().strVal();
			stack_lookup_table[ stack_pointer ].pop();
			if( bytecode[ c ][ 2 ] == 102 )
			{
				v2 = stack_lookup_table[ stack_pointer ].top().strVal();
				stack_lookup_table[ stack_pointer ].pop();
				stack_lookup_table[ stack_pointer ].push( ( * stack_lookup_table[ 1 ].top().chunkValue ).crunch( v.strVal(), v2.strVal() ) );
			}
			else
			{
				stack_lookup_table[ stack_pointer ].push( ( * stack_lookup_table[ 1 ].top().chunkValue ).crunch( v.strVal(), "" ) );
			}
			//delete (stack_lookup_table[ 1 ].top().chunkValue);
			stack_lookup_table[ 1 ].pop();
			break;
        case 23:
            // <top>
            merge_values( var_lookup_table[ bytecode[ c ][ 1 ] ], stack_lookup_table[ stack_pointer ].top(), bytecode[ c ][ 2 ], var_lookup_table[ bytecode[ c ][ 1 ] ] );
            break;
		case 27:
			// <push_dict>
			//stack_lookup_table[ stack_pointer ].top().setDict( var_lookup_table[ bytecode[ c ][ 1 ] ].strVal(), var_lookup_table[ bytecode[ c ][ 2 ] ] );
			break;
		case 28:
			// <pop_dict>
			//var_lookup_table[ bytecode[ c ][ 2 ] ] = stack_lookup_table[ stack_pointer ].top().getDict( var_lookup_table[ bytecode[ c ][ 1 ] ].strVal() );
			break;
		case 29:
			// <jump_mem>
			c = ( int ) ( var_lookup_table[ bytecode[ c ][ 2 ] ].intVal() );
			break;
		case 30:
			// <push_ip>
			stack_lookup_table[ stack_pointer ].push( variant( ( double ) c + 1 ) );
			break;
		case 31:
			// <call>
			// Same as a normal jump, but pushes the current IP onto the stack.
			
			// Recursion check.
			
			if( ( ( int ) stack_lookup_table[ stack_pointer ].size() ) > 10000 )
			{
				cout << "Too much recursion." << endl;
				return "Runtime error.";
			}
			
			stack_lookup_table[ stack_pointer ].push( variant( ( double ) c ) );
            c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            break;
		case 32:
			// <proc>
			break;
		case 33:
			// <endp>
			break;
		case 34:
			// <int>
			if ( bchasm_handle_softinterrupt( ( int ) var_lookup_table[ bytecode[ c ][ 2 ] ].intVal(), stack_pointer ) )
			{
				exit( 1 ); // We don't really have the kind of exception handling capability we need for this now.
			}
			break;
		case 35:
			// <fproc>
			break;
		case 36:
			// <fcall>
			// Same as a normal jump, but pushes the current IP onto the stack.
			
			// Recursion check.
			
			if( ( ( int ) stack_lookup_table[ stack_pointer ].size() ) > 10000 )
			{
				cout << "Too much recursion." << endl;
				return "Runtime error.";
			}
			
			stack_lookup_table[ stack_pointer ].push( variant( ( double ) c ) );
            c = label_lookup_table[ bytecode[ c ][ 2 ] ];
            break;
		case 37:
			// <cvar> || <check_variable>
			// Check the variable's dirty flag. If it's clean, set the variable to empty.
			if ( ! var_lookup_table[ bytecode[ c ][ 2 ] ].isDirty() )
			{
				var_lookup_table[ bytecode[ c ][ 2 ] ] = "";
			}
			break;
		case 38:
			// <push_link>
			// Creates an empty variant on the stack, and links it to the requested variant.
			stack_lookup_table[ stack_pointer ].push( variant( ) );
			stack_lookup_table[ stack_pointer ].top().linkTo( var_lookup_table[ bytecode[ c ][ 2 ] ] );
			break;
		case 39:
			// <forget>
			// Deletes the top item from the stack
			stack_lookup_table[ stack_pointer ].pop();
			break;
		case 97:
            // <hlt> || <halt>
            //return "Halted.";
			return var_lookup_table[ 0 ].strVal();
            break;
        case 98:
            // <wtf> || <wtf??>
            cout << "An error ocurred: " << var_lookup_table[ bytecode[ c ][ 2 ] ].strVal() << endl;
            return "Runtime error.";
            break;
		case -50:
            // <lbl> || <label>
            break;
        default:
			cout << "Illegal Instruction." << endl;
            return "Runtime error.";
        }
    }
    return var_lookup_table[ 0 ].strVal();
}

void bchasm::merge_values( variant &v1, variant &v2, int oper, variant &dest )
{
    switch ( oper )
    {
    case - 100:
        dest = v2;
        break;
    case -101:
        dest = v1.intVal() + v2.intVal();
        break;
    case - 102:
        dest = v1.intVal() - v2.intVal();
        break;
    case - 103:
        dest = v1.intVal() * v2.intVal();
        break;
    case - 104:
        dest = v1.intVal() / v2.intVal();
        break;
    case - 105:
        dest = v1.strVal() + v2.strVal();
        break;
    case - 106:
        dest = v1.strVal() + " " + v2.strVal();
        break;
    case - 107:
        dest = pow( v1.intVal(), v2.intVal() );
        break;
    case - 108:
        dest = sqrt( v1.intVal() );
        break;
    case -109:
        dest = bool_to_str( str_to_bool( v1.strVal() ) && str_to_bool( v2.strVal() ) );
        break;
    case - 110:
        dest = bool_to_str( str_to_bool( v1.strVal() ) || str_to_bool( v2.strVal() ) );
        break;
    case - 111:
        dest = bool_to_str( ( str_to_bool( v1.strVal() ) ^ str_to_bool( v2.strVal() ) ) );
        break;
    case - 112:
        dest = bool_to_str( !str_to_bool( v1.strVal() ) );
        break;
    case - 113:
        dest = bool_to_str( v1.intVal() > v2.intVal() );
        break;
    case - 114:
        dest = bool_to_str( v1.intVal() >= v2.intVal() );
        break;
    case - 115:
        dest = bool_to_str( v1.intVal() < v2.intVal() );
        break;
    case - 116:
        dest = bool_to_str( v1.intVal() <= v2.intVal() );
        break;
    case - 117:
        dest = bool_to_str( strcmpi( v1.strVal().c_str(), v2.strVal().c_str() ) == 0 );
        break;
    case - 118:
        dest = bool_to_str( strcmpi( v1.strVal().c_str(), v2.strVal().c_str() ) != 0 );
        break;
    case - 119:
        dest = 0 -v1.intVal();
        break;
	case - 120:
		dest = bool_to_str( instr( v1.strVal(), v2.strVal() ) >= 0 );
		break;
	case - 121:
		dest = bool_to_str( instr( v2.strVal(), v1.strVal() ) >= 0 );
		break;
	case -122:
		dest = bool_to_str( ! ( instr( v1.strVal(), v2.strVal() ) >= 0 ) );
		break;
	case -123:
		dest = bool_to_str( ! ( instr( v2.strVal(), v1.strVal() ) >= 0 ) );
		break;
	case -124:
        dest = ( double ) ( ( int ) v1.intVal() % ( int ) v2.intVal() );
        break;
    default:
        dest = v1;
    }
}

string bchasm::bool_to_str( bool b )
{
    return b ? "true" : "false";
}

bool bchasm::str_to_bool( const string &s )
{
    return ( strcmpi( s.c_str(), "true" ) == 0 ) ? true : false;
}

void bchasm::assemble( const string &program )
{

    /*

     bchasm::assemble(const string&) is a simple, finite state machine intended
     to convert CHASM2 input into bCHASM-compatible code. It strives to be case
     insensitive. The code is generated using vectors, so we can get the free
     dynamic memory on the fly, but the code is actually stored in a dynamically
     allocated two dimensional integer array (bchasm::bytecode).
     
    */

    int c, d, mode;						 // Generic counters and our state variable.

    vector < vector < int > > arsource;  // We assemble the program into a vector,
										 // and then copy it into an array before
										 // the assembler quits.

    vector < int > tempsource;			 // Stores an instruction-in-progress.

    initialize(); // Clears the memory maps and deallocates bytecode[].

    d = 0;
    mode = 3; // Ignore everything up to the first instruction.

    source = program;

    // Configure environment...
    fetch_identifier_index( ".result" );
    fetch_identifier_index( "stack main" );
    fetch_identifier_index( "stack chunks" );
    fetch_identifier_index( "stack objects" );
	
    string accumulator;
    for ( c = 0;c <= ( int ) program.size();c++ )
    {
        switch ( program[ c ] )
        {
        case '<':
            mode = 0;
            accumulator = "";
            break;
        case ' ':
            if ( mode == 0 )
            {
                if ( instr_lookup( accumulator ) == -50 )
                {
                    tempsource.push_back( -50 );
                }
                else
                {
                    tempsource.push_back( instr_lookup( accumulator ) );
                }
                mode = 1;
                accumulator = "";
            }
            else
            {
                accumulator += program[ c ];
            }
            break;
        case ',':
            tempsource.push_back( fetch_identifier_index( accumulator ) );
            mode = 2;
            accumulator = "";
            break;
        case '>':
            if ( tempsource[ 0 ] == -51 )
            {
                /* Skip it. It's a comment. */
                tempsource.clear();
            }
            else
            {
                if ( tempsource[ 0 ] == -50 )
                {
                    label_lookup_table[ fetch_identifier_index( accumulator ) ] = d;
                }
				else if ( ( tempsource[ 0 ] == 32 ) || ( tempsource[ 0 ] == 35 ) )
				{
                    label_lookup_table[ fetch_identifier_index( accumulator ) ] = d;
					cur_proc = accumulator;
				}
				else if ( tempsource[ 0 ] == 33 )
				{
					cur_proc = "";
				}
                tempsource.push_back( fetch_identifier_index( accumulator ) );
                arsource.push_back( tempsource );
                tempsource.pop_back();
                tempsource.pop_back();
                tempsource.pop_back();
                d++;
            }
            mode = 3;
            accumulator = "";
            break;
        default:
            accumulator += program[ c ];
        }
    }

    // Allocate the integer array...

    bytecode = new int * [ ( int ) arsource.size() ];

    for ( c = 0;c < ( int ) arsource.size();++c )
    {
        bytecode[ c ] = new int[ 3 ];
        bytecode[ c ][ 0 ] = arsource[ c ][ 0 ];
        bytecode[ c ][ 1 ] = arsource[ c ][ 1 ];
        bytecode[ c ][ 2 ] = arsource[ c ][ 2 ];
    }

    bytecode_length = ( int ) arsource.size();
	
    return ;
}

const string &bchasm::get_source( void )
{
    return source;
}

int *bchasm::get_bytecode( void )
{
    return * bytecode;
}

void bchasm::show_listing( void )
{
    int c;
    for ( c = 0;c < bytecode_length;++c )
    {
        cout << ( hex ) << setw( 8 ) << setfill( '0' ) << bytecode[ c ][ 0 ] << " "
        << ( hex ) << setw( 8 ) << setfill( '0' ) << bytecode[ c ][ 1 ] << " "
        << ( hex ) << setw( 8 ) << setfill( '0' ) << bytecode[ c ][ 2 ] << endl;
    }

    return ;
}

int bchasm::fetch_identifier_index( const string &fetch_line )
{
	
	// fetch_identifier_index parses a single "slot" (i.e. const 3, or label %endloop)
	// and returns an index for the appropriate array. If the desired element exists,
	// it returns the matching index, otherwise it will create one and return the
	// new index.
	
	// Allocate variables.
    string var_id = fetch_line;
    string var_id_orig = var_id;
    string r, r2;
    char tag[ 6 ];
    int i;

	
	// We need copies of the tag and the identifier, in both
	// mixed case and lowercase forms.
	
    strcpy( tag, var_id.substr( 0, 5 ).c_str() );

    for ( i = 0;i <= ( int ) var_id.length();i++ )
    {
        var_id[ i ] = tolower( var_id[ i ] );
    }

	
	// This is a brutal fix for the fact that the original CHASM2
	// specification didn't require any special tag for variables.
	// Since the introduction of the "world" and "local" tags, we
	// could probably remove this.
	
    if ( var_id.length() >= 7 )
    {
        r = var_id.substr( 6, ( int ) var_id.length() );
        r2 = var_id_orig.substr( 6, ( int ) var_id_orig.length() );
    }
    else
    {
        r = var_id;
        r2 = var_id;
    }

    if ( strcmpi( tag, "stack" ) == 0 )
    {
		// Search the stack index table and return the matching index
		// (or new one, as the case may be.

        if ( stack_lookup_indices.find( r ) != stack_lookup_indices.end() )
        {
			// Fetch the index and return it...
            return ( *stack_lookup_indices.find( r ) ).second;
        }
        else
        {
			// Create a new index and return that...
            stack_lookup_indices.insert( pair < string, int > ( r, ( int ) stack_lookup_indices.size() ) );
            stack_lookup_table.push_back( stack < variant > () );
            return ( int ) stack_lookup_indices.size() - 1;
        }

    }
    else if ( strcmpi( tag, "const" ) == 0 )
    {
		// Constants are indexed by their value. So, for example,
		// "const Hello World!" is indexed as "*CHello World!".
		
		// If we can find a matching key, we return the index,
		// otherwise we create a new key, populate it with the
		// constant value (as a string), and return that index.
		
        if ( var_lookup_indices.find( "*C" + r2 ) != var_lookup_indices.end() )
        {
			// Fetch the index and return it...
            return ( *var_lookup_indices.find( "*C" + r2 ) ).second;
        }
        else
        {
			// Create a new index and return that...
            var_lookup_indices.insert( pair < string, int > ( "*C" + r2, ( int ) var_lookup_indices.size() ) );
            var_lookup_table.push_back( quotesrecord::c2unescape( r2 ) );
            return ( int ) var_lookup_indices.size() - 1;
        }

    }
    else if ( strcmpi( tag, "label" ) == 0 )
    {
		// Labels are automatically created to point to the address -1.
		// They are populated by the assembler, when it finds a <LABEL>
		// instruction. In this way, label references without a matching
		// <LABEL> instruction can be easily identified by the interpreter.
		
		// Message passing banks on this.
		// DO YOU HEAR ME? DO NOT FOOK WITH THIS BEHAVIOR.
		
        if ( label_lookup_indices.find( r ) != label_lookup_indices.end() )
        {
			// Fetch the index and return it...
            return ( *label_lookup_indices.find( r ) ).second;
        }
        else
        {
			// Create a new index and return that...
            label_lookup_indices.insert( pair < string, int > ( r, ( int ) label_lookup_indices.size() ) );
			label_lookup_table.push_back( -1 );
            return ( int ) label_lookup_indices.size() - 1;
        }

    }
    else if ( strcmpi( tag, "class" ) == 0 )
    {
		// Classes (not to be confused with the OOP sense of the
		// term - they're similar but not the same) are essentially
		// constants defined by the assembler. The interpreter can
		// read the number and make decisions based on it. An example
		// would be the <MAKE> instruction. Here we just do a
		// case insensitive comparison and return the appropriate
		// number.
		
        if( strcmpi( r.c_str(), "chunk" ) == 0 )
		{
			return 25;
		}
		else if( ( strcmpi( r.c_str(), "char" ) == 0 ) || ( strcmpi( r.c_str(), "character" ) == 0 ) )
		{
			return 26;
		}
		else if( strcmpi( r.c_str(), "word" ) == 0 )
		{
			return 27;
		}
		else if( strcmpi( r.c_str(), "line" ) == 0 )
		{
			return 28;
		}
		else if( strcmpi( r.c_str(), "item" ) == 0 )
		{
			return 29;
		}
		else if( ( strcmpi( r.c_str(), "sent" ) == 0 ) || ( strcmpi( r.c_str(), "sentence" ) == 0 ) )
		{
			return 30;
		}
		else if( ( strcmpi( r.c_str(), "para" ) == 0 ) || ( strcmpi( r.c_str(), "paragraph" ) == 0 ) )
		{
			return 31;
		}
		else if( ( strcmpi( r.c_str(), "elem" ) == 0 ) || ( strcmpi( r.c_str(), "element" ) == 0 ) )
		{
			return 32;
		}
		else if( strcmpi( r.c_str(), "to" ) == 0 )
		{
			return 99;
		}
		else if( strcmpi( r.c_str(), "input" ) == 0 )
		{
			return 100;
		}
		else if( strcmpi( r.c_str(), "pair" ) == 0 )
		{
			return 101;
		}
		else if( strcmpi( r.c_str(), "output" ) == 0 )
		{
			return 102;
		}
		else
		{
			return 0;
		}
		
    }
    else if ( var_id[ 0 ] == '#' )
    {
		
		// "#" is a special kind of tag used to indicate
		// an inline operator. Each inline operator has
		// a specific code, so we just do a case insensitive
		// compare to look it up.
		
        if ( var_id == "#nop" )
        {
            return -100;
        }
        else if ( var_id == "#add" )
        {
            return -101;
        }
        else if ( var_id == "#sub" )
        {
            return -102;
        }
        else if ( var_id == "#mul" )
        {
            return -103;
        }
        else if ( var_id == "#div" )
        {
            return -104;
        }
        else if ( var_id == "#cat" )
        {
            return -105;
        }
        else if ( var_id == "#cct" )
        {
            return -106;
        }
        else if ( var_id == "#pow" )
        {
            return -107;
        }
        else if ( var_id == "#sqr" )
        {
            return -108;
        }
        else if ( var_id == "#and" )
        {
            return -109;
        }
        else if ( var_id == "#or " )
        {
            return -110;
        }
        else if ( var_id == "#xor" )
        {
            return -111;
        }
        else if ( var_id == "#not" )
        {
            return -112;
        }
        else if ( var_id == "#gtx" )
        {
            return -113;
        }
        else if ( var_id == "#gte" )
        {
            return -114;
        }
        else if ( var_id == "#ltx" )
        {
            return -115;
        }
        else if ( var_id == "#lte" )
        {
            return -116;
        }
        else if ( var_id == "#cmp" )
        {
            return -117;
        }
        else if ( var_id == "#neq" )
        {
            return -118;
        }
        else if ( var_id == "#neg" )
        {
            return -119;
        }
		else if ( var_id == "#has" )
		{
			return -120;
		}
		else if ( var_id == "#hs2" )
		{
			return -121;
		}
		else if ( var_id == "#hs3" )
		{
			return -122;
		}
		else if ( var_id == "#hs4" )
		{
			return -123;
		}
		else if ( var_id == "#mod" )
		{
			return -124;
		}
        else
        {
            return 0;
        }
    }
    else if ( strcmpi( tag, "world" ) == 0 )
    {
		
		// CHASM2 is aware of local variables, and prefixes them with
		// the name of their parent procedure, followed by "::". "global"
		// vars have no parent procedure, so they are prefixed with a
		// single "::".
		
        if ( var_lookup_indices.find( "::" + r2 ) != var_lookup_indices.end() )
        {
            return ( *var_lookup_indices.find( "::" + r2 ) ).second;
        }
        else
        {
            var_lookup_indices.insert( pair < string, int > ( "::" + r2, ( int ) var_lookup_indices.size() ) );
            var_lookup_table.push_back( r2 );
            //var_lookup_table.push_back( variant( "" ) );
            return ( int ) var_lookup_indices.size() - 1;
        }
    }
    else if ( strcmpi( tag, "local" ) == 0 )
    {
		
		// Local vars must be prefixed with the current procedure
		// name, which can be found in the data member "cur_proc".
		
		if ( var_lookup_indices.find( cur_proc + "::" + var_id ) != var_lookup_indices.end() )
		{
			return ( *var_lookup_indices.find( cur_proc + "::" + var_id ) ).second;
		}
		else
        {
			var_lookup_indices.insert( pair < string, int > ( cur_proc + "::" + var_id, ( int ) var_lookup_indices.size() ) );
			//var_lookup_table.push_back( fetch_line );
			var_lookup_table.push_back( r2 );
			//var_lookup_table.push_back( variant( "" ) );
			return ( int ) var_lookup_indices.size() - 1;
        }
    }
	
	// True and false are special constants. Please don't ask us why.
	// Really, we don't know anymore.
	
	else if ( var_id == "true")
	{
		return fetch_identifier_index( "const true" );
	}
	else if (var_id == "false" )
	{
		return fetch_identifier_index( "const false" );
	}
    else if ( var_id == "" )
    {
		// Um... yeah...
        return 0;
    }
    else if ( var_lookup_indices.find( cur_proc + "::" + var_id ) != var_lookup_indices.end() )
    {
		// No tag. We'll just assume it's a local var.
        return ( *var_lookup_indices.find( cur_proc + "::" + var_id ) ).second;
		
		// pxi_interface::loser_slap( * this_programmer ); // <--- Debug
    }
    else
    {
		// No tag. We'll just assume it's a local var.
		var_lookup_indices.insert( pair < string, int > ( cur_proc + "::" + var_id, ( int ) var_lookup_indices.size() ) );
        //var_lookup_table.push_back( fetch_line );
        var_lookup_table.push_back( r2 );
        //var_lookup_table.push_back( variant( "" ) );
        return ( int ) var_lookup_indices.size() - 1;

		// pxi_interface::loser_slap( * this_programmer ); // <--- Debug
    }
}

int bchasm::instr_lookup( const string& s1 )
{
    string s = s1;
    int i;
    for ( i = 0;i <= ( int ) s.length();i++ )
    {
        s[ i ] = tolower( s[ i ] );
    }
    if ( s == "mov" )
    {
        return 0;
    }
    else if ( s == "move" )
    {
        return 0;
    }
    else if ( s == "psh" )
    {
        return 1;
    }
    else if ( s == "push" )
    {
        return 1;
    }
    else if ( s == "pop" )
    {
        return 2;
    }
    else if ( s == "mrg" )
    {
        return 3;
    }
    else if ( s == "merge" )
    {
        return 3;
    }
    else if ( s == "swp" )
    {
        return 4;
    }
    else if ( s == "swap" )
    {
        return 4;
    }
    else if ( s == "scs" )
    {
        return 5;
    }
    else if ( s == "change_stack" )
    {
        return 5;
    }
    else if ( s == "add" )
    {
        return 6;
    }
    else if ( s == "sub" )
    {
        return 7;
    }
    else if ( s == "subtract" )
    {
        return 7;
    }
    else if ( s == "mul" )
    {
        return 8;
    }
    else if ( s == "multiply" )
    {
        return 8;
    }
    else if ( s == "jmp" )
    {
        return 9;
    }
    else if ( s == "jump" )
    {
        return 9;
    }
    else if ( s == "jmt" )
    {
        return 10;
    }
    else if ( s == "jump_if_mem_true" )
    {
        return 10;
    }
    else if ( s == "jmf" )
    {
        return 11;
    }
    else if ( s == "jump_if_mem_false" )
    {
        return 11;
    }
    else if ( s == "jmz" )
    {
        return 12;
    }
    else if ( s == "jump_if_mem_zero" )
    {
        return 12;
    }
    else if ( s == "jmn" )
    {
        return 13;
    }
    else if ( s == "jump_if_mem_nonzero" )
    {
        return 13;
    }
    else if ( s == "jst" )
    {
        return 14;
    }
    else if ( s == "jump_if_stack_true" )
    {
        return 14;
    }
    else if ( s == "jsf" )
    {
        return 15;
    }
    else if ( s == "jump_if_stack_false" )
    {
        return 15;
    }
    else if ( s == "jsz" )
    {
        return 16;
    }
    else if ( s == "jump_if_stack_zero" )
    {
        return 16;
    }
    else if ( s == "jsn" )
    {
        return 17;
    }
    else if ( s == "jump_if_stack_nonzero" )
    {
        return 17;
    }
    else if ( s == "jse" )
    {
        return 18;
    }
    else if ( s == "jump_if_stack_equals" )
    {
        return 18;
    }
    else if ( s == "jump_if_stack_notequals" )
    {
        return 26;
    }
    else if ( s == "nop" )
    {
        return 19;
    }
    else if ( s == "skip" )
    {
        return 19;
    }
    else if ( s == "top" )
    {
        return 23;
    }
    else if ( s == "wtf" )
    {
        return 98;
    }
    else if ( s == "wtf??" )
    {
        return 98;
    }
    else if ( s == "hlt" )
    {
        return 97;
    }
    else if ( s == "halt" )
    {
        return 97;
    }
    else if ( s == "lbl" )
    {
        return -50;
    }
    else if ( s == "label" )
    {
        return -50;
    }
    else if ( s == "!--" )
    {
        return -51;
    }
    else if ( s == "rem" )
    {
        return -51;
    }
    else if ( s == "//" )
    {
        return -51;
    }
	else if ( s == "make" )
	{
		return 20;
	}
	else if ( s == "push_dict" )
	{
		return 27;
	}
	else if ( s == "pop_dict" )
	{
		return 28;
	}
	else if ( s == "jump_mem" )
	{
		return 29;
	}
	else if ( s == "push_ip" )
	{
		return 30;
	}
	else if ( s == "call" )
	{
		return 31;
	}
	else if ( s == "proc" )
	{
		return 32;
	}
	else if ( s == "endp" )
	{
		return 33;
	}
	else if ( s == "int" )
	{
		return 34;
	}
	else if( s == "fproc" )
	{
		return 35;
	}
	else if( s == "fcall" )
	{
		return 36;
	}
	else if( s == "cvar" )
	{
		return 37;
	}
	else if( s == "check_variable" )
	{
		return 37;
	}
	else if( s == "push_link" )
	{
		return 38;
	}
	else if( s == "forget" )
	{
		return 39;
	}
	else if ( s == "chunk_add" )
	{
		return 21;
	}
	else if (s == "chunk_load" )
	{
		return 22;
	}
    else
    {
        return -1;
    };
}

int bchasm::strcmpi( const char* s1, const char* s2 )
{
    char c1, c2;
    while ( 1 )
    {
        c1 = tolower( *s1++ );
        c2 = tolower( *s2++ );
        if ( c1 < c2 )
            return -1;
        if ( c1 > c2 )
            return 1;
        if ( c1 == 0 )
            return 0;
    }
}

int bchasm::stricmp( const char* str1, const char* str2 )
{
    return strcmpi( str1, str2 );
}

void bchasm::assemble_to_file( const string & fname, const string & src )
{
	int c, l ( sizeof ( int ) );
	long size_counter ( 0 );
	
	// Declare a write-access binary stream...
	binarystream b( fname.c_str(), true, true );
	
	// Actually assemble the code...
	assemble( src );
	
	b.writeCString( "?CHASM" );
	// Now write the resulting configuration code, as a series of C-Strings...
	b.setPosition( 7 + l ); // Leave a double for the data length.
	
	// Write the variables.
	
	for( c = 0; c < ( int ) var_lookup_table.size(); ++c )
	{
		b.writeCString( var_lookup_table[ c ].strVal().c_str() );
		size_counter += var_lookup_table[ c ].strVal().size() + 1;
	}
	
	// Write the data length.
	b.setPosition( 7 );
	b.writeInt( size_counter );
	size_counter += l + 7;
	b.setPosition( size_counter );
	
	// Write out the label info.
	b.writeLong( ( ( int ) label_lookup_table.size() ) * l );
	
	for( c = 0; c < ( int ) label_lookup_table.size(); ++c )
	{
		b.writeInt( label_lookup_table[ c ] );
	}
	
	// Write out the stack info.
	b.writeInt( ( int ) stack_lookup_table.size() );
	b.writeInt( bytecode_length );
	
	// Write out the actual code.
	for( c = 0; c < bytecode_length; ++c )
	{
		b.writeInt( bytecode[ c ][ 0 ] );
		b.writeInt( bytecode[ c ][ 1 ] );
		b.writeInt( bytecode[ c ][ 2 ] );
	}
	
	// Close the file and exit.
	b.close();
	
	return;
}

string bchasm::interpret_from_file( const string & fname )
{
	string s;
	int i, size_counter( 0 );
	
	// Open the file for reading.
	binarystream b( fname.c_str(), true, false );
	
	s = b.readCStringAsString();
	
	if( s != "?CHASM" )
	{
		cout << "Not a valid CHASM2 binary." << endl;
		exit( 1 );
	}
	
	initialize();
	
	// Get the length of the data segment.
	i = b.readInt();
	
	// Read in the data segment.
	while( size_counter < i )
	{
		s = b.readCStringAsString();
		size_counter += ((int)s.size())+1;
		var_lookup_table.push_back( s );
	}
	size_counter = 0;
	
	// Get the length of the jump table.
	i = b.readInt();
	while( size_counter < i )
	{
		label_lookup_table.push_back( b.readInt() );
		size_counter += sizeof( int );
	}
	
	// Get the length of the stack table.
	i = b.readInt();
	
	for( ; i > 0; --i )
	{
		stack_lookup_table.push_back( stack< variant > () );
	}
	
	// Get the bytecode length.
	bytecode_length = b.readInt();
	
	i = 0;
	
    bytecode = new int * [ bytecode_length ];
	
	while( i < bytecode_length )
	{
		bytecode[ i ] = new int[ 3 ];
		bytecode[ i ][ 0 ] = b.readInt();
		bytecode[ i ][ 1 ] = b.readInt();
		bytecode[ i ][ 2 ] = b.readInt();
		++i;
	}
	
	b.close();
	
	// The system is initialized, now we can execute.
	
	interpret();
	
	return var_lookup_table[ 0 ].strVal();
}
